-- TRIGGERS (delimited with EOF)
DELIMITER EOF

-- Index an user when inserted or updated
create or replace function cy_fntg_index_user()
    returns trigger as $$
        begin
            NEW.name_tsvector := setweight(to_tsvector('unaccent', NEW.name), 'B');
            return NEW;
        end;
    $$ language plpgsql;
EOF

create trigger tg_index_user_insert 
    before insert on users
    for each row execute procedure cy_fntg_index_user();
EOF

create trigger tg_index_user_update 
    before update on users
    for each row when (
        old.name is distinct from new.name
    )
    execute procedure cy_fntg_index_user();
EOF

-- Index an advertisement when inserted or updated
create or replace function cy_fntg_index_ad()
    returns trigger as $$
        begin
            NEW.name_tsvector := setweight(to_tsvector('unaccent'::regconfig, NEW.name), 'A');
            NEW.description_tsvector := setweight(to_tsvector('unaccent'::regconfig, NEW.description), 'B');
            return NEW;
        end;
    $$ language plpgsql;
EOF

create trigger tg_index_ad_insert 
    before insert on ads
    for each row execute procedure cy_fntg_index_ad();
EOF

create trigger tg_index_ad_update 
    before update on ads
    for each row when (
        old.name is distinct from new.name
        or old.description is distinct from new.description
    )
    execute procedure cy_fntg_index_ad();
EOF

-- Index a custom field value when inserted or updated
create or replace function cy_fntg_index_custom_field_value()
    returns trigger as $$
        begin
            NEW.value_tsvector := to_tsvector('unaccent', coalesce(
                NEW.string_value, NEW.text_value, cy_strip_html_tags(NEW.rich_text_value), ''));
            return NEW;
        end;
    $$ language plpgsql;
EOF

create trigger tg_index_user_custom_field_value_insert
    before insert on user_custom_field_values
    for each row execute procedure cy_fntg_index_custom_field_value();
EOF

create trigger tg_index_user_custom_field_value_update 
    before update on user_custom_field_values
    for each row when (
        old.string_value is distinct from new.string_value
        or old.text_value is distinct from new.text_value
        or old.rich_text_value is distinct from new.rich_text_value
    )
    execute procedure cy_fntg_index_custom_field_value();
EOF

create trigger tg_index_contact_custom_field_value_insert
    before insert on contact_custom_field_values
    for each row execute procedure cy_fntg_index_custom_field_value();
EOF

create trigger tg_index_contact_custom_field_value_update 
    before update on contact_custom_field_values
    for each row when (
        old.string_value is distinct from new.string_value
        or old.text_value is distinct from new.text_value
        or old.rich_text_value is distinct from new.rich_text_value
    )
    execute procedure cy_fntg_index_custom_field_value();
EOF

create trigger tg_index_record_custom_field_value_insert
    before insert on record_custom_field_values
    for each row execute procedure cy_fntg_index_custom_field_value();
EOF

create trigger tg_index_record_custom_field_value_update 
    before update on record_custom_field_values
    for each row when (
        old.string_value is distinct from new.string_value
        or old.text_value is distinct from new.text_value
        or old.rich_text_value is distinct from new.rich_text_value
    )
    execute procedure cy_fntg_index_custom_field_value();
EOF

create trigger tg_index_ad_custom_field_value_insert
    before insert on ad_custom_field_values
    for each row execute procedure cy_fntg_index_custom_field_value();
EOF

create trigger tg_index_ad_custom_field_value_update 
    before update on ad_custom_field_values
    for each row when (
        old.string_value is distinct from new.string_value
        or old.text_value is distinct from new.text_value
        or old.rich_text_value is distinct from new.rich_text_value
    )
    execute procedure cy_fntg_index_custom_field_value();
EOF

-- Index an address when inserted or updated
create or replace function cy_fntg_index_address()
    returns trigger as $$
        begin
            NEW.full_tsvector = to_tsvector('unaccent'::regconfig, concat(
                NEW.address_line_1, ' ', NEW.address_line_2, ' ', NEW.street, ' ', NEW.building_number, ' ', NEW.complement, ' ',  
                NEW.city, ' ', NEW.neighborhood, ' ', NEW.po_box, ' ', NEW.region, ' ', NEW.zip));
            NEW.address_tsvector = to_tsvector('unaccent'::regconfig, concat(
                NEW.address_line_1, ' ', NEW.address_line_2, ' ', NEW.street, ' ', NEW.building_number, ' ', NEW.complement));
            NEW.neighborhood_tsvector = to_tsvector('unaccent'::regconfig, NEW.neighborhood);
            NEW.city_tsvector = to_tsvector('unaccent'::regconfig, NEW.city);
            NEW.region_tsvector = to_tsvector('unaccent'::regconfig, NEW.region);
            NEW.normalized_zip = lower(regexp_replace(NEW.zip, '\W', '', 'g'));
            NEW.normalized_po_box = lower(regexp_replace(NEW.po_box, '\W', '', 'g'));
            return NEW;
        end;
    $$ language plpgsql;
EOF

create trigger tg_index_address_insert
    before insert on addresses
    for each row execute procedure cy_fntg_index_address();
EOF

create trigger tg_index_address_update 
    before update on addresses
    for each row when (
        old.address_line_1 is distinct from new.address_line_1
        or old.address_line_2 is distinct from new.address_line_2
        or old.street is distinct from new.street
        or old.building_number is distinct from new.building_number
        or old.complement is distinct from new.complement
        or old.city is distinct from new.city
        or old.neighborhood is distinct from new.neighborhood
        or old.po_box is distinct from new.po_box
        or old.region is distinct from new.region
        or old.zip is distinct from new.zip
    )
    execute procedure cy_fntg_index_address();
EOF

create or replace function cy_fntg_create_geography()
	returns trigger as $$		
        begin
	        if NEW.longitude is null or NEW.latitude is null then
    			NEW.geography := null;
			else
    			NEW.geography := cy_to_geography(NEW.latitude, NEW.longitude);
			end if;	 		
			return NEW;
	    end
	$$ language plpgsql;
EOF

create trigger tg_create_geography_insert
	before insert on addresses
	for each row execute procedure cy_fntg_create_geography();
EOF

create trigger tg_create_geography_update 
    before update on addresses
    for each row when (
        old.latitude is distinct from new.latitude
        or old.longitude is distinct from new.longitude
    )
    execute procedure cy_fntg_create_geography();
EOF

-- Creates the background task executions for each incoming message inserted
create or replace function cy_fntg_insert_message_bg_task_execution()
	returns trigger as $$
		begin
			if (NEW.subclass = 'INCOMING') then
				insert into background_task_executions(class_name, context, priority, display) 
				values('org.cyclos.impl.messaging.MessageProcessingBackgroundTask', NEW.id, 1, concat('Process message with id ', NEW.id));
			end if;
			return NEW;
		end;
	$$ language plpgsql;
EOF

create trigger tg_create_message_bg_task_execution_insert
	after insert on messages
	for each row execute procedure cy_fntg_insert_message_bg_task_execution();
EOF

-- When a background task is finished and uses a fork_join, maybe finish it
create or replace function cy_fntg_background_tasks_fork_join()
    returns trigger as $$
        declare
            v_total_count integer;
            v_join_code text;
            v_finished_count integer;
        begin
            if OLD.fork_join_id is not null then
                insert into finished_fork_join_tasks(fork_join_id, task_id) 
                values(OLD.fork_join_id, OLD.id);
                
                select task_count, join_code
                from fork_joins
                where id = OLD.fork_join_id
                into v_total_count, v_join_code;
                
                select count(*)
                from finished_fork_join_tasks
                where fork_join_id = OLD.fork_join_id
                into v_finished_count;

                if v_finished_count >= v_total_count then
                    insert into background_task_executions
                    (class_name, context, priority, display)
                    values
                    ('org.cyclos.impl.system.InlineScriptBackgroundTask', 
                        v_join_code, 2 /* HIGH */, concat('Join code for ', OLD.fork_join_id));
                    
                    delete from finished_fork_join_tasks where fork_join_id = OLD.fork_join_id;
                    delete from fork_joins where id = OLD.fork_join_id;
                end if;
            end if;
            return NEW;
        end;
    $$ language plpgsql;
EOF

create trigger tg_background_tasks_fork_join
    after delete on background_task_executions
    for each row execute procedure cy_fntg_background_tasks_fork_join();
EOF

-- Revert the delimiter
DELIMITER ;
