-- VIEWS

-- Union of both owner and specific ad addresses
create view ads_all_addresses as	
	select a.id as address_id, a.ad_id from addresses a, ads ad where a.ad_id = ad.id and a.subclass = 'AD'
    union
	select address_id, ad_id from ads_addresses
	union
	select distinct dm.address_id, wdm.ad_id from 
		ad_delivery_methods dm, webshop_ads_delivery_methods wdm, ads ad where 
		dm.id = wdm.delivery_method_id and
		wdm.ad_id = ad.id and
		dm.address_id is not null;

-- Current user / broker relations 
create view users_brokers as
    select user_id, broker_id
    from brokerings
    where end_date is null;
-- As this is mapped to JPA, do nothing on delete from this view
create rule users_brokers_del as on delete to users_brokers
do instead nothing;

-- Current main broker of each user
create view user_main_broker as
    select user_id, broker_id
    from brokerings b
    where end_date is null
         and main_broker = true;
-- As this is mapped to JPA, do nothing on delete from this view
create rule user_main_broker_del as on delete to user_main_broker
    do instead nothing;

-- The last closed_account_balance per account
create view last_closed_account_balances as
	select
	    a.id as account_id,
	    b.id as closed_account_balance_id,
	    coalesce(b.date, a.archiving_date) as date,
	    coalesce(b.balance, a.archived_balance) as balance,
	    coalesce(b.reserved, a.archived_reserved) as reserved
	from accounts a
	left join lateral (
	    select *
	    from closed_account_balances b
	    where b.account_id = a.id
	    order by b.date desc
	    limit 1) b on true
	where coalesce(b.date, a.archiving_date) is not null;

-- Dirty balance amount sum per account
create view dirty_account_balances_diffs as
    select
        d.account_id,
        sum(case when t.from_id = d.account_id then -t.amount else t.amount end) as amount,
        array_agg(t.id) as dirty_transfer_ids
    from dirty_account_balances d inner join transfers t on d.transfer_id = t.id
    group by d.account_id;

-- Current cached account status per account
create view cached_account_status as
    select
        b.account_id,
        b.balance + coalesce(d.amount, 0) as balance,
        b.reserved,
        d.dirty_transfer_ids
    from account_balances b
        left join dirty_account_balances_diffs d on b.account_id = d.account_id;

-- Differences of either balance or reserved amount since the last account closing (see last_closed_account_balances)
create view account_status_diffs_since_closing as
	select diffs.account_id, sum(diffs.balance) as balance, sum(diffs.reserved) as reserved
	from (
	    select
	        t.from_id as account_id, -sum(t.amount) as balance,
	        0 as reserved
	    from transfers t
	        left join last_closed_account_balances c on t.from_id = c.account_id
	    where (c.date is null or t.date >= c.date)
	    group by t.from_id
	    union
	    select t.to_id, sum(t.amount), 0
	    from transfers t
	        left join last_closed_account_balances c on t.to_id = c.account_id
	    where (c.date is null or t.date >= c.date)
	    group by t.to_id
	    union
	    select r.account_id, 0, sum(r.amount)
	    from amount_reservations r
	        left join last_closed_account_balances c on r.account_id = c.account_id
	    where (c.date is null or r.date >= c.date)
	    group by r.account_id
	) diffs
	group by diffs.account_id;

-- The current account status per account from the last closing to summing up the transfers until now
-- No longer directly used in Cyclos, but left as an utility for db admins.
create view closed_account_status as
    select account_id, sum(balance) as balance, sum(reserved) as reserved
    from (
        select account_id, balance, reserved
        from last_closed_account_balances
        union all
        select account_id, balance, reserved
        from account_status_diffs_since_closing
    ) s
    group by account_id;
